define([
	'jQuery',
	'lodash',
	'angular',
	'app'
], function (
	$,
	_, 
	angular,
	app) {
	'use strict';
	app.service('assessmentService', function ($http, $q, localResourceDirectoryService, mhpuser, assessmentModels, escapeService) {

		var assessmentService = {};
		var assessmentAuthorizationUrl, assessmentsUrl, modifyAssessmentUrl, assessmentById, deleteAssessmentUrl, inactivateAssessmentUrl;

		function specialCharacters(assessment, escapeFunc) {
			assessment.uniqueTitle = escapeFunc(assessment.uniqueTitle);
			assessment.purpose = escapeFunc(assessment.purpose);
			assessment.notes = escapeFunc(assessment.notes);
			for(var index in assessment.question) {
				var question = assessment.question[index];
				question.prompt = escapeFunc(question.prompt);
				if(question.questionType == 'singleSelect' || question.questionType == 'multiSelect') {
					for(var choiceIndex = 0; choiceIndex < question.questionChoice.length; choiceIndex++) {
						var choice = question.questionChoice[choiceIndex].property[0]; 
						question.questionChoice[choiceIndex].choiceNumber = (choiceIndex + 1).toString();
						choice.value = escapeFunc(choice.value);
					}
				} else if(question.questionType == 'slider') {
					question.questionProperties[2].value = escapeFunc(question.questionProperties[2].value);
					question.questionProperties[3].value = escapeFunc(question.questionProperties[3].value);
				}
			}
		}

		function encodeAssessment(assessment) {
			assessment = angular.copy(assessment);
			for(var index in assessment.question) {
				var question = assessment.question[index];
				//Add question indicators
				question.id = "Q" + (parseInt(index) + 1);
				question.questionNumber = parseInt(index) + 1;
				//Strip unused question choices for each select
				if(question.questionType == 'singleSelect' || question.questionType == 'multiSelect') {
					for(var choiceIndex = question.questionChoice.length - 1; choiceIndex >= 0; choiceIndex--) {
						var choice = question.questionChoice[choiceIndex].property[0]; 
						if(choice.value == "") {
							question.questionChoice.splice(choiceIndex, 1);
						} 
					}
				}
			}
			specialCharacters(assessment, escapeService.escape);
			return assessment;
		}

		function decodeAssessment(assessment) {
			assessment.purpose = assessment.purpose ? assessment.purpose : "";
			assessment.waitingPeriod = assessment.waitingPeriod ? assessment.waitingPeriod : 0; 
			
			_.each(assessment.question, function(question) {
				//Readd types when the response comes back
				if(question.type == 'FREE_TEXT') {
					question.questionType = 'text';
				} else if(question.type == 'SLIDER') {
					question.questionType = 'slider';
				} else if(question.type == 'MULTI_OPTION_MULTI_RESPONSE') {
					question.questionType = 'multiSelect';
				} else if(question.type == 'MULTI_OPTION_SINGLE_RESPONSE') {
					if(	question.questionChoice.length == 2 && 
						question.questionChoice[0].property[0].value == 'Yes' && 
						question.questionChoice[1].property[0].value == 'No') {
						question.questionType = 'yesNo';
					} else {
						question.questionType = 'singleSelect';
					}
				}
				//Readd extra question choices that were originally stripped
				if(question.questionType == 'singleSelect' || question.questionType == 'multiSelect') {
					for(var index = question.questionChoice.length; index < 6; index++) {
						var emptyQuestionChoice = angular.copy(assessmentModels['emptyQuestionChoice']);
						emptyQuestionChoice.choiceNumber = (index + 1).toString();
						question.questionChoice.push(emptyQuestionChoice);
					}
				}
			}); 
			specialCharacters(assessment, escapeService.unescape);
		}

		assessmentService.readLocalResourceDirectory = function() {
			var deferred = $q.defer();
			localResourceDirectoryService.fetch().then(function(directory) {
				assessmentAuthorizationUrl = directory['accessment-builder-authorization'].replace('{site-code}', mhpuser.vistaLocation).replace('{ien}', mhpuser.id);
				assessmentsUrl = directory['accessment-builder-assessments'];
				assessmentById = directory['accessment-builder-assessments-by-id'];
				modifyAssessmentUrl = directory['accessment-builder-modify-assessment'].replace('{site-code}', mhpuser.vistaLocation).replace('{ien}', mhpuser.id);
				deleteAssessmentUrl = directory['accessment-builder-delete-assessment'].replace('{site-code}', mhpuser.vistaLocation).replace('{ien}', mhpuser.id);
				inactivateAssessmentUrl = directory['accessment-builder-inactivate-assessment'].replace('{site-code}', mhpuser.vistaLocation).replace('{ien}', mhpuser.id);
				deferred.resolve();
			});
			return deferred.promise;
		};

		var isAuthorized = false;

        assessmentService.isAuthorized = function () {
            return isAuthorized;
        };

		assessmentService.authorize = function() {
			var deferred = $q.defer();
			assessmentService.readLocalResourceDirectory().then(function(response) {
					$http.get(assessmentAuthorizationUrl).then(function(response) {
                        isAuthorized = response.data === 'true';

                        deferred.resolve(isAuthorized);
					}, function(reason) {
						deferred.reject(reason);
					})
				}
			);
			return deferred.promise;
		};

		assessmentService.getBlankAssessment = function() {
			return angular.copy(assessmentModels['blankAssessment']);
		}; 

		assessmentService.getAssessments = function() {
			var config = {};
			config.transformResponse =  function(data, headersGetter, status) {
				data = JSON.parse(data); 
				_.each(data.assessment, function(assessment) {
					decodeAssessment(assessment); 
					assessment.lastModified = assessment.lastModifiedDate.split(' ')[0];
				}); 
				return data; 
			}
			return $http.get(assessmentsUrl, config);
		};

		assessmentService.getAssessmentById = function(id) {
			var config = {};
			config.transformResponse =  function(data, headersGetter, status) {
				data = JSON.parse(data);
				decodeAssessment(data);
				return data; 
			}
			return $http.get(assessmentById + id, config);
		};

		assessmentService.saveAssessment = function(assessment) {
			var config = {};
			config.transformResponse =  function(data, headersGetter, status) {
				data = JSON.parse(data);
				if(data.errorType != 400) {
					decodeAssessment(data);
				}
				return data; 
			};

			if(assessment.id) {
				return $http.put(modifyAssessmentUrl, encodeAssessment(assessment), config);
			} else {
				return $http.post(modifyAssessmentUrl, encodeAssessment(assessment), config);
			}
		};

		assessmentService.getModel = function(type) {
			var model = angular.copy(assessmentModels[type]);
			if(type == 'singleSelect' || type == 'multiSelect') {
				for(var i = 1; i <= 6; i++) {
					var questionChoice = angular.copy(assessmentModels['emptyQuestionChoice']);
					questionChoice.choiceNumber = i.toString();
					model.questionChoice.push(questionChoice);
				}
			}
			return model;
		}; 

		assessmentService.deleteAssessment = function(id) {
			return $http.delete(deleteAssessmentUrl + id); 
		};

		assessmentService.inactivateAssessment = function(id) {
			return $http.put(inactivateAssessmentUrl + id)
		};

		assessmentService.modifyAssessment = function(assessment) {
			assessment = angular.copy(assessment); 
			delete assessment.id; 
			assessment.version += 1; 
			assessment.status = "Draft";
			assessment.publish = false; 
			return this.saveAssessment(assessment);
		};

		return assessmentService;
	});
});
